/* global sc */

let $$ = require('./debug')({ debug: 'cache', path: module.id });  
$$.obj = require('./obj');

/**
 * Extracts the package name and root directory from a given directory path.
 * @param {string} dirname - The directory path.
 * @returns {Object} - An object containing the package name and root directory.
 */
const packageFromDirName = dirname => {
  let dirnames = dirname.split('/');
  let index = dirnames.indexOf('customers');
  return {
    name: dirnames[index + 1],
    root: dirnames.slice(0, index + 2).join('/'),
  };
};

/**
 * Creates a package object based on the provided reference.
 *
 * @param {number|string|object} ref - The reference used to create the package object.
 * @returns {object|null} The package object created based on the reference, or null if the reference is invalid.
 */
module.exports = (ref) => {
  if (ref === null || ref === undefined) return;
  let id, path;
  if (typeof ref === 'number') {
    id = ref; // numeric object id
  } else if (typeof ref === 'string') {
    // argument is a string value
    if (!isNaN(ref)) {
      id = parseInt(ref); // id string from module
    } else if ($$.obj.isUUID(ref)) {
      id = ref; // file/folder UUID
    } else {
      // argument is a file path or package name
      path = ref.startsWith('/') ? ref : '/agorum/roi/customers/' + ref;
    }
  } else {
    // argument is an object
    if (ref.exports && ref.id) {
      if (!isNaN(ref.id)) {
        id = parseInt(ref.id); // id from module
      } else if (ref.filename) {
        path = ref.filename; // file path
      } else {
        throw new Error('not implemented');
      }
    }
  }
  if (id && typeof process === 'undefined') {
    let objects = require('common/objects').sc($$.environment.context || sc);
    path = objects.find(id).anyFolderPath.split('/').slice(0, -1).join('/');
  }
  if (path) {
    return packageFromDirName(path);
  }
  return null;
};

